//
//  Copyright 2010-2011 Justin.tv. All rights reserved.
//

#import "TWHUDView.h"
#import "UIFont+Twitch.h"

static BOOL showingHUD = NO;

@implementation TWHUDView

#define HUDSide 160.f
#define HUDSideBoundary 15.f

#define HUDShowAnimationDuration .2
#define HUDHideAnimationDuration .2

#define HUDOnScreenDurationAnimation 4.

CGAffineTransform transformForOrientation() {
	if ([UIApplication sharedApplication].statusBarOrientation == UIInterfaceOrientationPortraitUpsideDown)
		return CGAffineTransformMakeRotation(M_PI);
	
	if ([UIApplication sharedApplication].statusBarOrientation == UIInterfaceOrientationLandscapeLeft)
		return CGAffineTransformMakeRotation(M_PI_2);
	
	if ([UIApplication sharedApplication].statusBarOrientation == UIInterfaceOrientationLandscapeRight)
		return CGAffineTransformMakeRotation(-M_PI_2);
	
	// always default to the identity transform, but, it is also what is returned for portrait orientation
	return CGAffineTransformIdentity;
}

+ (NSTimeInterval) onScreenDuration {
	return HUDOnScreenDurationAnimation;
}

#pragma mark -

- (id) initWithCenterView:(UIView *) centerView labelText:(NSString *) labelText {
	[[UIDevice currentDevice] beginGeneratingDeviceOrientationNotifications];
	
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(deviceDidRotate:) name:UIDeviceOrientationDidChangeNotification object:nil];
	
	CGFloat screenWidth = [UIScreen mainScreen].bounds.size.width;
	CGFloat screenHeight = [UIScreen mainScreen].bounds.size.height;
	
	CGRect frame = CGRectMake((screenWidth / 2.f) - (HUDSide / 2.f), (screenHeight / 2.f) - (HUDSide / 2.f), HUDSide, HUDSide);
	
	if (!(self = [super initWithFrame:frame]))
		return nil;
	
	_textLabel = [[UILabel alloc] initWithFrame:CGRectZero];
	_textLabel.font = [UIFont twitchHUDTextFont];
	_textLabel.numberOfLines = 1;
	_textLabel.textAlignment = NSTextAlignmentCenter;
	_textLabel.lineBreakMode = NSLineBreakByWordWrapping;
	_textLabel.backgroundColor = [UIColor clearColor];
	_textLabel.textColor = [UIColor whiteColor];
	_textLabel.text = labelText;
	_textLabel.hidden = YES;
	
	_centerView = centerView;
	_centerView.hidden = YES;
	
	[self addSubview:_textLabel];
	[self addSubview:_centerView];
	
	self.hidden = YES;
	self.alpha = 0.f;
	
	return self;
}

- (id) initWithCenterView:(UIView *) centerView {
	return [self initWithCenterView:centerView labelText:nil];
}

- (void) dealloc {
	[[UIDevice currentDevice] endGeneratingDeviceOrientationNotifications];
	
	[NSObject cancelPreviousPerformRequestsWithTarget:self selector:@selector(removeFromSuperviewAnimated) object:nil];
	[NSObject cancelPreviousPerformRequestsWithTarget:self selector:@selector(show) object:nil];
	
	[[NSNotificationCenter defaultCenter] removeObserver:self name:UIDeviceOrientationDidChangeNotification object:nil];
}

#pragma mark -

+ (TWHUDView *) activityIndicatorHUDView {
	UIActivityIndicatorView *activityIndicatorView = [[UIActivityIndicatorView alloc] initWithActivityIndicatorStyle:UIActivityIndicatorViewStyleWhiteLarge];
	[activityIndicatorView startAnimating];
	
	TWHUDView *hudView = [[[self class] alloc] initWithCenterView:activityIndicatorView];
	
	return hudView;
}

+ (void) _showHUDWithImage:(UIImage *) image labelText:(NSString *) labelText {
	if (showingHUD)
		return;
	
	showingHUD = YES;
	
	UIImageView *imageView = [[UIImageView alloc] initWithImage:image];
	TWHUDView *hudView = [[TWHUDView alloc] initWithCenterView:imageView labelText:labelText];
	[hudView show];
	[hudView performSelector:@selector(removeFromSuperviewAnimated) withObject:nil afterDelay:HUDOnScreenDurationAnimation];
}

+ (void) showSuccessHUD {
	[self showSuccessHUDWithText:TKLocalizedString(@"Success", @"Success HUD text")];
}

+ (void) showSuccessHUDWithText:(NSString *) text {
	[self _showHUDWithImage:[UIImage imageFromFrameworkBundleNamed:@"alert_overlay_check"] labelText:text];
}

+ (void) showErrorHUD {
	[self showErrorHUDWithText:TKLocalizedString(@"Error", @"Error HUD text")];
}

+ (void) showErrorHUDWithText:(NSString *) text {
	[self _showHUDWithImage:[UIImage imageFromFrameworkBundleNamed:@"alert_overlay_x"] labelText:text];
}

#pragma mark -

- (void) _show {
	_textLabel.hidden = NO;
	_centerView.hidden = NO;
}

- (void) show {
	if (!self.hidden)
		return;
	
	[[UIApplication sharedApplication].keyWindow addSubview:self];
	
	self.hidden = NO;
	self.layer.transform = CATransform3DMakeScale(2.f, 2.f, 1.f);
	
	[UIView animateWithDuration:HUDShowAnimationDuration delay:0. options:UIViewAnimationOptionCurveEaseOut animations:^{
		
		_textLabel.hidden = NO;
		_centerView.hidden = NO;
		self.alpha = 1.f;
		self.layer.transform = CATransform3DMakeScale(1.f, 1.f, 1.f);
		
	}completion:^(BOOL finished) {
		
	}];
}

#pragma mark -

- (void) layoutSubviews {
	self.layer.cornerRadius = 10.f;
	self.backgroundColor = [UIColor colorWithRed:0.f green:0.f blue:0.f alpha:.75f];
	
	CGRect frame = _centerView.frame;
	CGRect contentFrame = self.frame;
	
	CGSize textSize = CGSizeZero;
	CGSize contentSize = CGSizeMake(contentFrame.size.width - (HUDSideBoundary * 2.f), contentFrame.size.height - (HUDSideBoundary * 2.f));
	
	if (_textLabel.text) {
		
		textSize = [_textLabel sizeThatFits:contentSize];
		_textLabel.frame = CGRectMake(floorf((contentFrame.size.width / 2.f) - (textSize.width / 2.f)) , floorf(contentFrame.size.height - (HUDSideBoundary / 2.f) - textSize.height), textSize.width, textSize.height);
		
		[_textLabel sizeToFit];
	}
	_centerView.frame = CGRectMake(floorf((contentFrame.size.width / 2.f) - (frame.size.width / 2.f)), floorf(((contentFrame.size.height / 2.f) - (frame.size.height / 2.f))), frame.size.width, frame.size.height);
	
	frame = _textLabel.frame;
	frame.size.width = self.frame.size.width;
	frame.origin.x = 0.;
	_textLabel.frame = frame;
	
	// when displaying in landscape or portraitUpsideDown orientation
	self.transform = transformForOrientation();
}

#pragma mark -

- (void) removeFromSuperviewAnimated:(BOOL) animated {
	showingHUD = NO;
	
	if (!animated) {
		[self removeFromSuperview];
		return;
	}
	
	[UIView animateWithDuration:HUDHideAnimationDuration delay:0. options:UIViewAnimationOptionCurveEaseIn animations:^{
		
		self.alpha = 0.f;
	}completion:^(BOOL finished) {
		
		self.hidden = YES;
		[self removeFromSuperview];
	}];
}

- (void) removeFromSuperviewAnimated {
	[self removeFromSuperviewAnimated:YES];
}

#pragma mark -

- (void) deviceDidRotate:(NSNotification *) notification {
	[UIView animateWithDuration:[UIApplication sharedApplication].statusBarOrientationAnimationDuration animations:^{
		self.transform = transformForOrientation();
	}];
}
@end
